<?php
/**
 * Plugin Name: Modern Football Brackets Live (panel + AJAX)
 * Description: Llaves de fútbol con base de datos de equipos y estadios, panel rápido de resultados, cierre manual de partidos y actualización en vivo sin recargar la página.
 * Version: 1.5.2
 * Author: ChatGPT + Ignacio
 * Text Domain: mfb-live-panel-ajax
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MFB_Live_Panel_Ajax_V7 {

    public function __construct() {
        add_action( 'init', array( $this, 'register_cpts' ) );
        add_action( 'add_meta_boxes', array( $this, 'add_bracket_metabox' ) );
        add_action( 'save_post', array( $this, 'save_bracket_meta' ) );
        add_action( 'admin_menu', array( $this, 'register_results_page' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        add_action( 'wp_ajax_mfb_get_bracket_html', array( $this, 'ajax_get_bracket_html' ) );
        add_action( 'wp_ajax_nopriv_mfb_get_bracket_html', array( $this, 'ajax_get_bracket_html' ) );

        add_shortcode( 'football_bracket', array( $this, 'shortcode_bracket' ) );
        // 👉 nuevo shortcode para panel frontal
    add_shortcode( 'football_results_panel', array( $this, 'shortcode_results_panel' ) );
    
    }

    /* =========================
     *  CPTs
     * =======================*/
    public function register_cpts() {
        register_post_type(
            'mfb_bracket',
            array(
                'labels' => array(
                    'name'          => __( 'Llaves de Fútbol', 'mfb-live-panel-ajax' ),
                    'singular_name' => __( 'Llave de Fútbol', 'mfb-live-panel-ajax' ),
                    'add_new_item'  => __( 'Añadir nueva llave', 'mfb-live-panel-ajax' ),
                    'edit_item'     => __( 'Editar llave', 'mfb-live-panel-ajax' ),
                    'menu_name'     => __( 'Llaves de Fútbol', 'mfb-live-panel-ajax' ),
                ),
                'public'       => false,
                'show_ui'      => true,
                'show_in_menu' => true,
                'menu_icon'    => 'dashicons-sos',
                'supports'     => array( 'title' ),
            )
        );

        register_post_type(
            'mfb_team',
            array(
                'labels' => array(
                    'name'          => __( 'Equipos', 'mfb-live-panel-ajax' ),
                    'singular_name' => __( 'Equipo', 'mfb-live-panel-ajax' ),
                    'add_new_item'  => __( 'Añadir nuevo equipo', 'mfb-live-panel-ajax' ),
                    'edit_item'     => __( 'Editar equipo', 'mfb-live-panel-ajax' ),
                    'menu_name'     => __( 'Equipos', 'mfb-live-panel-ajax' ),
                ),
                'public'       => false,
                'show_ui'      => true,
                'show_in_menu' => true,
                'menu_icon'    => 'dashicons-groups',
                'supports'     => array( 'title', 'thumbnail' ),
            )
        );

        register_post_type(
            'mfb_stadium',
            array(
                'labels' => array(
                    'name'          => __( 'Estadios', 'mfb-live-panel-ajax' ),
                    'singular_name' => __( 'Estadio', 'mfb-live-panel-ajax' ),
                    'add_new_item'  => __( 'Añadir nuevo estadio', 'mfb-live-panel-ajax' ),
                    'edit_item'     => __( 'Editar estadio', 'mfb-live-panel-ajax' ),
                    'menu_name'     => __( 'Estadios', 'mfb-live-panel-ajax' ),
                ),
                'public'       => false,
                'show_ui'      => true,
                'show_in_menu' => true,
                'menu_icon'    => 'dashicons-location-alt',
                'supports'     => array( 'title' ),
            )
        );
    }

    /* =========================
     *  Assets públicos
     * =======================*/
    public function enqueue_assets() {
        wp_register_style(
            'mfb-bracket-style',
            plugin_dir_url( __FILE__ ) . 'assets/css/bracket.css',
            array(),
            '1.5.2'
        );

        wp_register_script(
            'mfb-bracket-live',
            plugin_dir_url( __FILE__ ) . 'assets/js/bracket-live.js',
            array(),
            '1.5.2',
            true
        );

        wp_localize_script(
            'mfb-bracket-live',
            'mfbLive',
            array(
                'ajax_url'         => admin_url( 'admin-ajax.php' ),
                'refresh_interval' => 8000,
            )
        );
    }

    /* =========================
     *  Metabox configuración
     * =======================*/
    public function add_bracket_metabox() {
        add_meta_box(
            'mfb_bracket_data',
            __( 'Configuración de la llave', 'mfb-live-panel-ajax' ),
            array( $this, 'render_bracket_metabox' ),
            'mfb_bracket',
            'normal',
            'high'
        );
    }

    private function generate_empty_bracket( $teams_count, $post_id ) {
    $id_prefix = 'b' . intval( $post_id ) . '-m';
        $teams_count = intval( $teams_count );
        if ( ! in_array( $teams_count, array( 2, 4, 8, 16 ), true ) ) {
    $teams_count = 4;
}
        $round_names_2  = array( 'Final' );
$round_names_4  = array( 'Semifinal', 'Final' );
$round_names_8  = array( 'Cuartos de final', 'Semifinal', 'Final' );
$round_names_16 = array( 'Octavos de final', 'Cuartos de final', 'Semifinal', 'Final' );

switch ( $teams_count ) {
    case 16:
        $round_names = $round_names_16;
        break;
    case 8:
        $round_names = $round_names_8;
        break;
    case 4:
        $round_names = $round_names_4;
        break;
    case 2:
    default:
        $round_names = $round_names_2;
        break;
}


        $rounds       = array();
        $match_id     = 1;
        $previous_ids = array();

        foreach ( $round_names as $round_index => $round_name ) {
            $matches_in_round = $teams_count / pow( 2, $round_index + 1 );
            $round_matches    = array();
            $current_ids      = array();

            for ( $m = 0; $m < $matches_in_round; $m++ ) {
                $id            = $id_prefix . $match_id . $match_id;
                $match_id++;
                $current_ids[] = $id;

                $match = array(
                    'id'             => $id,
                    'score_regular'  => array( 'home' => null, 'away' => null ),
                    'went_penalties' => false,
                    'finished'       => false,
                );

                if ( 0 !== $round_index ) {
                    $prev_home_index    = $m * 2;
                    $prev_away_index    = $m * 2 + 1;
                    $match['home_from'] = $previous_ids[ $prev_home_index ];
                    $match['away_from'] = $previous_ids[ $prev_away_index ];
                }

                $round_matches[] = $match;
            }

            $rounds[]     = array(
                'name'    => $round_name,
                'matches' => $round_matches,
            );
            $previous_ids = $current_ids;
        }

        return array(
            'teams_count' => $teams_count,
            'rounds'      => $rounds,
        );
    }

    public function render_bracket_metabox( $post ) {
        wp_nonce_field( 'mfb_save_bracket', 'mfb_bracket_nonce' );

        $json = get_post_meta( $post->ID, '_mfb_bracket_json', true );
        $data = json_decode( $json, true );
        $has_data = is_array( $data ) && ! empty( $data['rounds'] );

        $teams = get_posts(
            array(
                'post_type'   => 'mfb_team',
                'numberposts' => -1,
                'post_status' => 'publish',
                'orderby'     => 'title',
                'order'       => 'ASC',
            )
        );

        $stadiums = get_posts(
            array(
                'post_type'   => 'mfb_stadium',
                'numberposts' => -1,
                'post_status' => 'publish',
                'orderby'     => 'title',
                'order'       => 'ASC',
            )
        );

        echo '<div class="mfb-admin-wrapper">';
        echo '<p><strong>Paso 1:</strong> Crea equipos en <strong>Equipos</strong> (con logo) y estadios en <strong>Estadios</strong>.</p>';
        echo '<p><strong>Paso 2:</strong> Aquí armas las llaves, defines fecha/hora, estadio y marcador.</p>';

        if ( ! $has_data ) {
            ?>
            <p><strong>¿Con cuántos equipos quieres iniciar esta llave?</strong></p>
            <p>
                <label><input type="radio" name="mfb_new_teams" value="2"> 2 equipos (solo final)</label><br>
<label><input type="radio" name="mfb_new_teams" value="4" checked> 4 equipos</label><br>
<label><input type="radio" name="mfb_new_teams" value="8"> 8 equipos</label><br>
<label><input type="radio" name="mfb_new_teams" value="16"> 16 equipos</label><br>

            </p>
            <p><em>Guarda la entrada para que se generen automáticamente los partidos.</em></p>
            <?php
            echo '</div>';
            return;
        }

        $rounds = $data['rounds'];

        // Calculamos ganadores para mostrar nombres reales cuando se pueda
        $winners = $this->calculate_winners_from_rounds( $rounds );

        echo '<hr><h3>Partidos por ronda</h3>';
        echo '<div class="mfb-admin-rounds">';

        foreach ( $rounds as $ri => $round ) {
            $round_name = isset( $round['name'] ) ? $round['name'] : 'Ronda ' . ( $ri + 1 );
            echo '<div class="mfb-admin-round">';
            printf(
                '<h4>Ronda %d: <input type="text" name="mfb_rounds[%d][name]" value="%s" style="width:60%%;"></h4>',
                $ri + 1,
                $ri,
                esc_attr( $round_name )
            );

            if ( ! empty( $round['matches'] ) && is_array( $round['matches'] ) ) {
                echo '<div class="mfb-admin-matches">';

                foreach ( $round['matches'] as $mi => $match ) {
                    $match_id = isset( $match['id'] ) ? $match['id'] : $id_prefix . $match_id . ( $ri + 1 ) . '_' . ( $mi + 1 );

                    $home_from = isset( $match['home_from'] ) ? $match['home_from'] : '';
                    $away_from = isset( $match['away_from'] ) ? $match['away_from'] : '';

                    $home_team_id = isset( $match['home_team_id'] ) ? intval( $match['home_team_id'] ) : 0;
                    $away_team_id = isset( $match['away_team_id'] ) ? intval( $match['away_team_id'] ) : 0;

                    $sr_home = isset( $match['score_regular']['home'] ) ? $match['score_regular']['home'] : '';
                    $sr_away = isset( $match['score_regular']['away'] ) ? $match['score_regular']['away'] : '';

                    $went_penalties = ! empty( $match['went_penalties'] );
                    $pen_home       = isset( $match['penalties']['home'] ) ? $match['penalties']['home'] : '';
                    $pen_away       = isset( $match['penalties']['away'] ) ? $match['penalties']['away'] : '';

                    $date = isset( $match['date'] ) ? $match['date'] : '';
                    $time = isset( $match['time'] ) ? $match['time'] : '';
                    $stadium_id = isset( $match['stadium_id'] ) ? intval( $match['stadium_id'] ) : 0;

                    $finished = ! empty( $match['finished'] );

                    echo '<div class="mfb-admin-match-card">';
                    printf(
                        '<input type="hidden" name="mfb_rounds[%1$d][matches][%2$d][id]" value="%3$s">',
                        $ri,
                        $mi,
                        esc_attr( $match_id )
                    );
                    printf(
                        '<input type="hidden" name="mfb_rounds[%1$d][matches][%2$d][home_from]" value="%3$s">',
                        $ri,
                        $mi,
                        esc_attr( $home_from )
                    );
                    printf(
                        '<input type="hidden" name="mfb_rounds[%1$d][matches][%2$d][away_from]" value="%3$s">',
                        $ri,
                        $mi,
                        esc_attr( $away_from )
                    );

                    echo '<div class="mfb-admin-match-header">Partido ' . esc_html( $match_id ) . '</div>';

                    echo '<div class="mfb-admin-team-row"><div class="mfb-admin-team-side"><strong>Local</strong><br>';
                    if ( $home_from ) {
                        if ( isset( $winners[ $home_from ] ) ) {
                            echo '<em>' . esc_html( $winners[ $home_from ]['name'] ) . ' (ganador de ' . esc_html( $home_from ) . ')</em>';
                        } else {
                            echo '<em>Ganador de ' . esc_html( $home_from ) . '</em>';
                        }
                    } else {
                        printf(
                            '<select name="mfb_rounds[%1$d][matches][%2$d][home_team_id]" style="width:100%%;">',
                            $ri,
                            $mi
                        );
                        echo '<option value="0">-- Selecciona equipo --</option>';
                        foreach ( $teams as $team ) {
                            printf(
                                '<option value="%d" %s>%s</option>',
                                $team->ID,
                                selected( $home_team_id, $team->ID, false ),
                                esc_html( $team->post_title )
                            );
                        }
                        echo '</select>';
                    }
                    echo '</div>';
                    echo '<div class="mfb-admin-team-score">';
                    printf(
                        '<label>Goles<br><input type="number" min="0" name="mfb_rounds[%1$d][matches][%2$d][score_home]" value="%3$s" style="width:60px;"></label>',
                        $ri,
                        $mi,
                        esc_attr( $sr_home )
                    );
                    echo '</div></div>';

                    echo '<div class="mfb-admin-team-row"><div class="mfb-admin-team-side"><strong>Visita</strong><br>';
                    if ( $away_from ) {
                        if ( isset( $winners[ $away_from ] ) ) {
                            echo '<em>' . esc_html( $winners[ $away_from ]['name'] ) . ' (ganador de ' . esc_html( $away_from ) . ')</em>';
                        } else {
                            echo '<em>Ganador de ' . esc_html( $away_from ) . '</em>';
                        }
                    } else {
                        printf(
                            '<select name="mfb_rounds[%1$d][matches][%2$d][away_team_id]" style="width:100%%;">',
                            $ri,
                            $mi
                        );
                        echo '<option value="0">-- Selecciona equipo --</option>';
                        foreach ( $teams as $team ) {
                            printf(
                                '<option value="%d" %s>%s</option>',
                                $team->ID,
                                selected( $away_team_id, $team->ID, false ),
                                esc_html( $team->post_title )
                            );
                        }
                        echo '</select>';
                    }
                    echo '</div>';
                    echo '<div class="mfb-admin-team-score">';
                    printf(
                        '<label>Goles<br><input type="number" min="0" name="mfb_rounds[%1$d][matches][%2$d][score_away]" value="%3$s" style="width:60px;"></label>',
                        $ri,
                        $mi,
                        esc_attr( $sr_away )
                    );
                    echo '</div></div>';

                    echo '<div class="mfb-admin-datetime-row">';
                    printf(
                        '<label>Fecha<br><input type="date" name="mfb_rounds[%1$d][matches][%2$d][date]" value="%3$s"></label>',
                        $ri,
                        $mi,
                        esc_attr( $date )
                    );
                    printf(
                        '<label>Hora<br><input type="time" name="mfb_rounds[%1$d][matches][%2$d][time]" value="%3$s"></label>',
                        $ri,
                        $mi,
                        esc_attr( $time )
                    );
                    echo '<label>Estadio<br>';
                    printf(
                        '<select name="mfb_rounds[%1$d][matches][%2$d][stadium_id]">',
                        $ri,
                        $mi
                    );
                    echo '<option value="0">-- Sin estadio --</option>';
                    foreach ( $stadiums as $stadium ) {
                        printf(
                            '<option value="%d" %s>%s</option>',
                            $stadium->ID,
                            selected( $stadium_id, $stadium->ID, false ),
                            esc_html( $stadium->post_title )
                        );
                    }
                    echo '</select></label>';
                    echo '</div>';

                    echo '<div class="mfb-admin-penalties-row">';
                    printf(
                        '<label><input type="checkbox" name="mfb_rounds[%1$d][matches][%2$d][went_penalties]" value="1" %3$s> Definido por penales</label>',
                        $ri,
                        $mi,
                        checked( $went_penalties, true, false )
                    );
                    echo '<div class="mfb-admin-penalty-inputs">';
                    printf(
                        '<label>Penales L <input type="number" min="0" name="mfb_rounds[%1$d][matches][%2$d][pen_home]" value="%3$s" style="width:60px;"></label>',
                        $ri,
                        $mi,
                        esc_attr( $pen_home )
                    );
                    printf(
                        '<label>Penales V <input type="number" min="0" name="mfb_rounds[%1$d][matches][%2$d][pen_away]" value="%3$s" style="width:60px;"></label>',
                        $ri,
                        $mi,
                        esc_attr( $pen_away )
                    );
                    echo '</div>';
                    echo '<div class="mfb-admin-finished-flag">';
                    printf(
                        '<label><input type="checkbox" name="mfb_rounds[%1$d][matches][%2$d][finished]" value="1" %3$s> Partido finalizado</label>',
                        $ri,
                        $mi,
                        checked( $finished, true, false )
                    );
                    echo '</div>';
                    echo '</div>';

                    echo '</div>';
                }

                echo '</div>';
            }

            echo '</div>';
        }

        echo '</div>';
        echo '</div>';
    }

    public function save_bracket_meta( $post_id ) {
        if ( ! isset( $_POST['mfb_bracket_nonce'] ) || ! wp_verify_nonce( $_POST['mfb_bracket_nonce'], 'mfb_save_bracket' ) ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( isset( $_POST['post_type'] ) && 'mfb_bracket' === $_POST['post_type'] ) {
            if ( ! current_user_can( 'edit_post', $post_id ) ) {
                return;
            }
        }

        $existing_json = get_post_meta( $post_id, '_mfb_bracket_json', true );
        $existing_data = json_decode( $existing_json, true );
        $has_data      = is_array( $existing_data ) && ! empty( $existing_data['rounds'] );

        if ( ! $has_data && isset( $_POST['mfb_new_teams'] ) ) {
            $teams_count = intval( $_POST['mfb_new_teams'] );
            $data = $this->generate_empty_bracket( $teams_count, $post_id );
            update_post_meta( $post_id, '_mfb_bracket_json', wp_json_encode( $data ) );
            return;
        }

        if ( isset( $_POST['mfb_rounds'] ) && is_array( $_POST['mfb_rounds'] ) ) {
            $posted_rounds = wp_unslash( $_POST['mfb_rounds'] );
            $new_rounds    = array();

            foreach ( $posted_rounds as $ri => $round ) {
                $round_name = isset( $round['name'] ) ? sanitize_text_field( $round['name'] ) : 'Ronda';
                $matches    = array();

                if ( isset( $round['matches'] ) && is_array( $round['matches'] ) ) {
                    foreach ( $round['matches'] as $mi => $match ) {
                        $id        = isset( $match['id'] ) ? sanitize_text_field( $match['id'] ) : $id_prefix . $match_id . ( $ri + 1 ) . '_' . ( $mi + 1 );
                        $home_from = isset( $match['home_from'] ) ? sanitize_text_field( $match['home_from'] ) : '';
                        $away_from = isset( $match['away_from'] ) ? sanitize_text_field( $match['away_from'] ) : '';

                        $home_team_id = isset( $match['home_team_id'] ) ? intval( $match['home_team_id'] ) : 0;
                        $away_team_id = isset( $match['away_team_id'] ) ? intval( $match['away_team_id'] ) : 0;

                        $score_home = isset( $match['score_home'] ) && $match['score_home'] !== '' ? intval( $match['score_home'] ) : null;
                        $score_away = isset( $match['score_away'] ) && $match['score_away'] !== '' ? intval( $match['score_away'] ) : null;

                        $went_penalties = isset( $match['went_penalties'] ) && '1' === $match['went_penalties'];

                        $pen_home = isset( $match['pen_home'] ) && $match['pen_home'] !== '' ? intval( $match['pen_home'] ) : null;
                        $pen_away = isset( $match['pen_away'] ) && $match['pen_away'] !== '' ? intval( $match['pen_away'] ) : null;

                        $date = isset( $match['date'] ) ? sanitize_text_field( $match['date'] ) : '';
                        $time = isset( $match['time'] ) ? sanitize_text_field( $match['time'] ) : '';

                        $stadium_id = isset( $match['stadium_id'] ) ? intval( $match['stadium_id'] ) : 0;

                        $finished = isset( $match['finished'] ) && '1' === $match['finished'];

                        $match_arr = array(
                            'id'             => $id,
                            'score_regular'  => array(
                                'home' => $score_home,
                                'away' => $score_away,
                            ),
                            'went_penalties' => $went_penalties,
                            'finished'       => $finished,
                        );

                        if ( null !== $pen_home && null !== $pen_away ) {
                            $match_arr['penalties'] = array(
                                'home' => $pen_home,
                                'away' => $pen_away,
                            );
                        }

                        if ( $date || $time ) {
                            $match_arr['date'] = $date;
                            $match_arr['time'] = $time;
                        }

                        if ( $stadium_id > 0 ) {
                            $match_arr['stadium_id'] = $stadium_id;
                        }

                        if ( $home_from ) {
                            $match_arr['home_from'] = $home_from;
                        } elseif ( $home_team_id > 0 ) {
                            $match_arr['home_team_id'] = $home_team_id;
                        }

                        if ( $away_from ) {
                            $match_arr['away_from'] = $away_from;
                        } elseif ( $away_team_id > 0 ) {
                            $match_arr['away_team_id'] = $away_team_id;
                        }

                        $matches[] = $match_arr;
                    }
                }

                $new_rounds[] = array(
                    'name'    => $round_name,
                    'matches' => $matches,
                );
            }

            $data = array( 'rounds' => $new_rounds );
            update_post_meta( $post_id, '_mfb_bracket_json', wp_json_encode( $data ) );
        }
    }

    /* =========================
     *  Panel rápido resultados
     * =======================*/
    public function register_results_page() {
        add_submenu_page(
            'edit.php?post_type=mfb_bracket',
            __( 'Resultados en vivo', 'mfb-live-panel-ajax' ),
            __( 'Resultados en vivo', 'mfb-live-panel-ajax' ),
            'edit_posts',
            'mfb-live-results',
            array( $this, 'render_results_page' )
        );
    }

    private function get_team_data( $team_id ) {
        $team_id = intval( $team_id );
        if ( $team_id <= 0 ) {
            return null;
        }

        $post = get_post( $team_id );
        if ( ! $post || 'mfb_team' !== $post->post_type ) {
            return null;
        }

        $name = get_the_title( $team_id );
        $logo = get_the_post_thumbnail_url( $team_id, 'thumbnail' );

        return array(
            'id'   => $team_id,
            'name' => $name,
            'logo' => $logo,
        );
    }

    public function render_results_page() {
        // Eliminamos la restricción de capacidad para que funcione en el front-end
        // if ( ! current_user_can( 'edit_posts' ) ) {
        //     return;
        // }


        if ( isset( $_POST['mfb_results_nonce'] ) && wp_verify_nonce( $_POST['mfb_results_nonce'], 'mfb_save_results' ) ) {
            $bracket_id = isset( $_POST['mfb_selected_bracket'] ) ? intval( $_POST['mfb_selected_bracket'] ) : 0;
            $scores     = isset( $_POST['scores'] ) && is_array( $_POST['scores'] ) ? $_POST['scores'] : array();

            if ( $bracket_id && ! empty( $scores ) ) {
                $json = get_post_meta( $bracket_id, '_mfb_bracket_json', true );
                $data = json_decode( $json, true );

                if ( is_array( $data ) && ! empty( $data['rounds'] ) ) {
                    foreach ( $data['rounds'] as $ri => $round ) {
                        if ( empty( $round['matches'] ) || ! is_array( $round['matches'] ) ) {
                            continue;
                        }
                        foreach ( $round['matches'] as $mi => $match ) {
                            $match_id = isset( $match['id'] ) ? $match['id'] : '';
                            if ( ! $match_id || ! isset( $scores[ $match_id ] ) ) {
                                continue;
                            }
                            $row = $scores[ $match_id ];

                            $score_home = isset( $row['home'] ) && $row['home'] !== '' ? intval( $row['home'] ) : null;
                            $score_away = isset( $row['away'] ) && $row['away'] !== '' ? intval( $row['away'] ) : null;

                            $went_penalties = isset( $row['went_penalties'] ) && '1' === $row['went_penalties'];

                            $pen_home = isset( $row['pen_home'] ) && $row['pen_home'] !== '' ? intval( $row['pen_home'] ) : null;
                            $pen_away = isset( $row['pen_away'] ) && $row['pen_away'] !== '' ? intval( $row['pen_away'] ) : null;

                            $finished = isset( $row['finished'] ) && '1' === $row['finished'];
                            $scorers_home = isset( $row['scorers_home'] ) ? sanitize_textarea_field( $row['scorers_home'] ) : '';
$scorers_away = isset( $row['scorers_away'] ) ? sanitize_textarea_field( $row['scorers_away'] ) : '';


// Guardamos / limpiamos goleadores local
if ( '' !== $scorers_home ) {
    $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_home'] = $scorers_home;
} elseif ( isset( $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_home'] ) ) {
    unset( $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_home'] );
}

// Guardamos / limpiamos goleadores visita
if ( '' !== $scorers_away ) {
    $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_away'] = $scorers_away;
} elseif ( isset( $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_away'] ) ) {
    unset( $data['rounds'][ $ri ]['matches'][ $mi ]['scorers_away'] );
}


                            $data['rounds'][ $ri ]['matches'][ $mi ]['score_regular']['home'] = $score_home;
                            $data['rounds'][ $ri ]['matches'][ $mi ]['score_regular']['away'] = $score_away;
                            $data['rounds'][ $ri ]['matches'][ $mi ]['went_penalties']        = $went_penalties;
                            $data['rounds'][ $ri ]['matches'][ $mi ]['finished']              = $finished;

                            if ( null !== $pen_home && null !== $pen_away ) {
                                $data['rounds'][ $ri ]['matches'][ $mi ]['penalties'] = array(
                                    'home' => $pen_home,
                                    'away' => $pen_away,
                                );
                            } else {
                                if ( isset( $data['rounds'][ $ri ]['matches'][ $mi ]['penalties'] ) ) {
                                    unset( $data['rounds'][ $ri ]['matches'][ $mi ]['penalties'] );
                                }
                            }
                        }
                    }

                    update_post_meta( $bracket_id, '_mfb_bracket_json', wp_json_encode( $data ) );
                    echo '<div class="updated notice"><p>Resultados actualizados correctamente.</p></div>';
                }
            }
        }

        $brackets = get_posts(
            array(
                'post_type'   => 'mfb_bracket',
                'numberposts' => -1,
                'post_status' => 'any',
                'orderby'     => 'title',
                'order'       => 'ASC',
            )
        );

        $selected_id = isset( $_POST['mfb_selected_bracket'] ) ? intval( $_POST['mfb_selected_bracket'] ) : 0;
        if ( ! $selected_id && isset( $_GET['bracket_id'] ) ) {
            $selected_id = intval( $_GET['bracket_id'] );
        }

        echo '<div class="wrap">';
        echo '<h1>Resultados en vivo</h1>';
        echo '<p>Elige una llave y actualiza el marcador rápidamente. Marca <strong>Partido finalizado</strong> solo cuando quieras que el ganador pase a la siguiente ronda.</p>';

        echo '<form method="post">';
        wp_nonce_field( 'mfb_save_results', 'mfb_results_nonce' );

        echo '<p><label for="mfb_selected_bracket"><strong>Llave / torneo:</strong> </label> ';
        echo '<select id="mfb_selected_bracket" name="mfb_selected_bracket" onchange="this.form.submit()">';
        echo '<option value="0">-- Selecciona una llave --</option>';
        foreach ( $brackets as $bracket ) {
            printf(
                '<option value="%d" %s>%s</option>',
                $bracket->ID,
                selected( $selected_id, $bracket->ID, false ),
                esc_html( $bracket->post_title )
            );
        }
        echo '</select></p>';

        if ( $selected_id ) {
            $json = get_post_meta( $selected_id, '_mfb_bracket_json', true );
            $data = json_decode( $json, true );

            
        if ( is_array( $data ) && ! empty( $data['rounds'] ) ) {

            // Calculamos ganadores para poder mostrar los nombres reales en semifinales / final
            $winners = $this->calculate_winners_from_rounds( $data['rounds'] );

            echo '<hr>';
            echo '<table class="widefat fixed striped">';
            echo '<thead><tr>';
            echo '<th>Ronda</th>';
echo '<th>ID partido</th>';
echo '<th>Local</th>';
echo '<th>Visita</th>';
echo '<th style="width:140px;">Goles (L - V)</th>';
echo '<th style="width:180px;">Goleadores local</th>';
echo '<th style="width:180px;">Goleadores visita</th>';
echo '<th style="width:180px;">Penales (L - V)</th>';
echo '<th style="width:80px;">¿Penales?</th>';
echo '<th style="width:120px;">Finalizado</th>';
            echo '</tr></thead><tbody>';

            foreach ( $data['rounds'] as $ri => $round ) {
                $round_name = isset( $round['name'] ) ? $round['name'] : 'Ronda ' . ( $ri + 1 );

                if ( empty( $round['matches'] ) || ! is_array( $round['matches'] ) ) {
                    continue;
                }

                foreach ( $round['matches'] as $mi => $match ) {
                    $match_id = isset( $match['id'] ) ? $match['id'] : $id_prefix . $match_id . ( $ri + 1 ) . '_' . ( $mi + 1 );

                    // Etiqueta de local
                    $home_team_label = 'Por definir';
                    if ( isset( $match['home_team_id'] ) ) {
                        $team = $this->get_team_data( $match['home_team_id'] );
                        if ( $team ) {
                            $home_team_label = $team['name'];
                        }
                    } elseif ( isset( $match['home_from'] ) ) {
                        if ( isset( $winners[ $match['home_from'] ] ) ) {
                            $home_team_label = $winners[ $match['home_from'] ]['name'];
                        } else {
                            $home_team_label = 'Ganador de ' . $match['home_from'];
                        }
                    }

                    // Etiqueta de visita
                    $away_team_label = 'Por definir';
                    if ( isset( $match['away_team_id'] ) ) {
                        $team = $this->get_team_data( $match['away_team_id'] );
                        if ( $team ) {
                            $away_team_label = $team['name'];
                        }
                    } elseif ( isset( $match['away_from'] ) ) {
                        if ( isset( $winners[ $match['away_from'] ] ) ) {
                            $away_team_label = $winners[ $match['away_from'] ]['name'];
                        } else {
                            $away_team_label = 'Ganador de ' . $match['away_from'];
                        }
                    }

                    $sr_home = isset( $match['score_regular']['home'] ) ? $match['score_regular']['home'] : '';
                    $sr_away = isset( $match['score_regular']['away'] ) ? $match['score_regular']['away'] : '';

                    $went_penalties = ! empty( $match['went_penalties'] );
                    $pen_home       = isset( $match['penalties']['home'] ) ? $match['penalties']['home'] : '';
                    $pen_away       = isset( $match['penalties']['away'] ) ? $match['penalties']['away'] : '';

                    $finished = ! empty( $match['finished'] );
                    $scorers_home = isset( $match['scorers_home'] ) ? $match['scorers_home'] : '';
$scorers_away = isset( $match['scorers_away'] ) ? $match['scorers_away'] : '';


                    echo '<tr>';
                    echo '<td>' . esc_html( $round_name ) . '</td>';
                    echo '<td>' . esc_html( $match_id ) . '</td>';
                    echo '<td>' . esc_html( $home_team_label ) . '</td>';
                    echo '<td>' . esc_html( $away_team_label ) . '</td>';

                    // Goles
echo '<td>';
printf(
    '<input type="number" name="scores[%1$s][home]" value="%2$s" min="0" style="width:60px;"> - <input type="number" name="scores[%1$s][away]" value="%3$s" min="0" style="width:60px;">',
    esc_attr( $match_id ),
    esc_attr( $sr_home ),
    esc_attr( $sr_away )
);
echo '</td>';

// Goleadores local
echo '<td>';
printf(
    '<input type="text" name="scores[%1$s][scorers_home]" value="%2$s" style="width:100%%;" placeholder="Ej: 2x Pérez, 1x Soto">',
    esc_attr( $match_id ),
    esc_attr( $scorers_home )
);
echo '</td>';

// Goleadores visita
echo '<td>';
printf(
    '<input type="text" name="scores[%1$s][scorers_away]" value="%2$s" style="width:100%%;" placeholder="Ej: 1x López">',
    esc_attr( $match_id ),
    esc_attr( $scorers_away )
);
echo '</td>';

// Penales
echo '<td>';
printf(
    '<input type="number" name="scores[%1$s][pen_home]" value="%2$s" min="0" style="width:60px;"> - <input type="number" name="scores[%1$s][pen_away]" value="%3$s" min="0" style="width:60px;">',
    esc_attr( $match_id ),
    esc_attr( $pen_home ),
    esc_attr( $pen_away )
);
echo '</td>';


                    echo '<td>';
                    printf(
                        '<label><input type="checkbox" name="scores[%1$s][went_penalties]" value="1" %2$s> Sí</label>',
                        esc_attr( $match_id ),
                        checked( $went_penalties, true, false )
                    );
                    echo '</td>';

                    echo '<td>';
                    printf(
                        '<label><input type="checkbox" name="scores[%1$s][finished]" value="1" %2$s> Partido finalizado</label>',
                        esc_attr( $match_id ),
                        checked( $finished, true, false )
                    );
                    echo '</td>';

                    echo '</tr>';
                }
            }

            echo '</tbody></table>';
            echo '<p class="submit"><button type="submit" class="button-primary">Guardar resultados</button></p>';
        } else {
            echo '<p>No hay estructura de llaves para esta entrada.</p>';
        }

            
        }

        echo '</form>';
        echo '</div>';
    }
public function shortcode_results_panel( $atts ) {

    // Usamos la misma hoja de estilos del bracket para dar look moderno
    if ( function_exists( 'wp_enqueue_style' ) ) {
        wp_enqueue_style( 'mfb-bracket-style' );
    }

    ob_start();

    // Contenedor especial para el panel en el front-end
    echo '<div class="mfb-results-frontend">';
    echo '  <div class="mfb-results-scroll">';

    // Reutilizamos exactamente el mismo panel del admin
    $this->render_results_page();

    echo '  </div>';
    echo '</div>';

    return ob_get_clean();
}



    /* Helpers fecha / estadio */
    private function get_stadium_name( $stadium_id ) {
        $stadium_id = intval( $stadium_id );
        if ( $stadium_id <= 0 ) {
            return '';
        }
        $post = get_post( $stadium_id );
        if ( ! $post || 'mfb_stadium' !== $post->post_type ) {
            return '';
        }
        return get_the_title( $stadium_id );
    }

    private function format_datetime_label( $date, $time ) {
        if ( ! $date ) {
            return '';
        }
        $time_str = $time ? $time : '00:00';
        $ts       = strtotime( $date . ' ' . $time_str );
        if ( $ts ) {
            $label = date_i18n( 'd-m-Y', $ts );
        } else {
            $label = $date;
        }
        if ( $time ) {
            $label .= ' · ' . $time . ' hrs';
        }
        return $label;
    }

    /* =========================
     *  Render HTML del bracket
     * =======================*/
    private function render_bracket_html( $post_id ) {
        $post_id = intval( $post_id );
        if ( ! $post_id ) {
            return '';
        }

        $json = get_post_meta( $post_id, '_mfb_bracket_json', true );
        if ( empty( $json ) ) {
            return '<p>No hay datos de llave configurados.</p>';
        }

        $data = json_decode( $json, true );
        if ( ! is_array( $data ) || empty( $data['rounds'] ) ) {
            return '<p>Datos de llave inválidos.</p>';
        }

        $rounds  = $data['rounds'];
        $winners = array();
        $now     = current_time( 'timestamp' );

        foreach ( $rounds as $ri => $round ) {
            if ( empty( $round['matches'] ) || ! is_array( $round['matches'] ) ) {
                continue;
            }

            foreach ( $round['matches'] as $mi => $match ) {

                if ( isset( $match['home_from'] ) && isset( $winners[ $match['home_from'] ] ) ) {
                    $rounds[ $ri ]['matches'][ $mi ]['home_team'] = $winners[ $match['home_from'] ];
                } elseif ( isset( $match['home_team_id'] ) ) {
                    $team = $this->get_team_data( $match['home_team_id'] );
                    if ( $team ) {
                        $rounds[ $ri ]['matches'][ $mi ]['home_team'] = $team;
                    }
                }

                if ( isset( $match['away_from'] ) && isset( $winners[ $match['away_from'] ] ) ) {
                    $rounds[ $ri ]['matches'][ $mi ]['away_team'] = $winners[ $match['away_from'] ];
                } elseif ( isset( $match['away_team_id'] ) ) {
                    $team = $this->get_team_data( $match['away_team_id'] );
                    if ( $team ) {
                        $rounds[ $ri ]['matches'][ $mi ]['away_team'] = $team;
                    }
                }

                $match = $rounds[ $ri ]['matches'][ $mi ];

                $match_id  = isset( $match['id'] ) ? $match['id'] : null;
                $home_team = isset( $match['home_team'] ) ? $match['home_team'] : null;
                $away_team = isset( $match['away_team'] ) ? $match['away_team'] : null;

                $sr_home = isset( $match['score_regular']['home'] ) ? intval( $match['score_regular']['home'] ) : null;
                $sr_away = isset( $match['score_regular']['away'] ) ? intval( $match['score_regular']['away'] ) : null;

                $went_penalties = ! empty( $match['went_penalties'] );
                $pen_home       = isset( $match['penalties']['home'] ) ? intval( $match['penalties']['home'] ) : null;
                $pen_away       = isset( $match['penalties']['away'] ) ? intval( $match['penalties']['away'] ) : null;

                $finished    = ! empty( $match['finished'] );
                $stadium_id  = isset( $match['stadium_id'] ) ? intval( $match['stadium_id'] ) : 0;
                $stadium_name = $stadium_id ? $this->get_stadium_name( $stadium_id ) : '';
                


                $winner = null;
                if ( $finished && $home_team && $away_team && null !== $sr_home && null !== $sr_away ) {
                    if ( $sr_home > $sr_away ) {
                        $winner = $home_team;
                    } elseif ( $sr_away > $sr_home ) {
                        $winner = $away_team;
                    } elseif ( $went_penalties && null !== $pen_home && null !== $pen_away ) {
                        if ( $pen_home > $pen_away ) {
                            $winner = $home_team;
                        } elseif ( $pen_away > $pen_home ) {
                            $winner = $away_team;
                        }
                    }
                }

                if ( $match_id && $winner && $finished ) {
                    $winners[ $match_id ] = $winner;
                }

                $rounds[ $ri ]['matches'][ $mi ]['stadium_name'] = $stadium_name;
            }
        }

        ob_start();
        ?>
        <div class="mfb-bracket">
            <?php foreach ( $rounds as $ri => $round ) : ?>
    <?php $is_final_round = ( $ri === count( $rounds ) - 1 ); ?>
    <div class="mfb-round">
        <div class="mfb-round-title">
            <?php echo esc_html( isset( $round['name'] ) ? $round['name'] : 'Ronda ' . ( $ri + 1 ) ); ?>
        </div>
        <div class="mfb-round-matches">
            <?php if ( ! empty( $round['matches'] ) && is_array( $round['matches'] ) ) : ?>
                <?php foreach ( $round['matches'] as $mi => $match ) : ?>

                                <?php
                                $match_id  = isset( $match['id'] ) ? $match['id'] : null;
                                $home_team = isset( $match['home_team'] ) ? $match['home_team'] : null;
                                $away_team = isset( $match['away_team'] ) ? $match['away_team'] : null;

                                $sr_home = isset( $match['score_regular']['home'] ) ? intval( $match['score_regular']['home'] ) : null;
                                $sr_away = isset( $match['score_regular']['away'] ) ? intval( $match['score_regular']['away'] ) : null;

                                $went_penalties = ! empty( $match['went_penalties'] );
                                $pen_home       = isset( $match['penalties']['home'] ) ? intval( $match['penalties']['home'] ) : null;
                                $pen_away       = isset( $match['penalties']['away'] ) ? intval( $match['penalties']['away'] ) : null;
    $scorers_home = isset( $match['scorers_home'] ) ? $match['scorers_home'] : '';
    $scorers_away = isset( $match['scorers_away'] ) ? $match['scorers_away'] : '';

                                $winner      = ( $match_id && isset( $winners[ $match_id ] ) ) ? $winners[ $match_id ] : null;
                                $home_is_win = $winner && $home_team && ( $winner['name'] === $home_team['name'] );
                                $away_is_win = $winner && $away_team && ( $winner['name'] === $away_team['name'] );

                                $date = isset( $match['date'] ) ? $match['date'] : '';
                                $time = isset( $match['time'] ) ? $match['time'] : '';
$scorers_home = isset( $match['scorers_home'] ) ? $match['scorers_home'] : '';
$scorers_away = isset( $match['scorers_away'] ) ? $match['scorers_away'] : '';

$scorers_home_lines = array_filter(
    array_map(
        'trim',
        preg_split( '/[\r\n]+|,|;/', $scorers_home )
    )
);

$scorers_away_lines = array_filter(
    array_map(
        'trim',
        preg_split( '/[\r\n]+|,|;/', $scorers_away )
    )
);


                                $date_time_str = $this->format_datetime_label( $date, $time );
                                $stadium_name  = isset( $match['stadium_name'] ) ? $match['stadium_name'] : '';

                                $finished = ! empty( $match['finished'] );

                                $is_live = false;
                                if ( $date ) {
                                    $time_str = $time ? $time : '00:00';
                                    $ts       = strtotime( $date . ' ' . $time_str );
                                    if ( $ts ) {
                                        $diff = $now - $ts;
                                        if ( $diff >= 0 && $diff <= 3 * HOUR_IN_SECONDS && ! $finished ) {
                                            $is_live = true;
                                        }
                                    }
                                }

                                $score_class_main = 'mfb-score-main';
                                $score_class_pen  = 'mfb-score-pen';
                                if ( $is_live && ! $went_penalties ) {
                                    $score_class_main .= ' mfb-score-live';
                                } elseif ( $is_live && $went_penalties ) {
                                    $score_class_main .= ' mfb-score-regular-live';
                                    $score_class_pen  .= ' mfb-score-pen-live';
                                }
                                ?>
                                <div class="mfb-match">
                                    <div class="mfb-team-row <?php echo $home_is_win ? 'mfb-winner' : ''; ?>">
                                        <div class="mfb-team">
                                            <div class="mfb-team-logo">
                                                <?php if ( $home_team && ! empty( $home_team['logo'] ) ) : ?>
                                                    <img src="<?php echo esc_url( $home_team['logo'] ); ?>" alt="<?php echo esc_attr( $home_team['name'] ); ?>">
                                                <?php endif; ?>
                                            </div>
                                            <div class="mfb-team-name">
    <?php echo esc_html( $home_team ? $home_team['name'] : 'Por definir' ); ?>

    <?php if ( ! empty( $scorers_home_lines ) ) : ?>
        <div class="mfb-scorers-block">
            <?php foreach ( $scorers_home_lines as $line ) : ?>
                <div class="mfb-scorers-line">
                    <span class="mfb-scorer-ball">⚽</span>
                    <span class="mfb-scorer-text"><?php echo esc_html( $line ); ?></span>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <?php if ( $is_final_round && $finished && $home_is_win ) : ?>
        <div class="mfb-champion-label">🏆 CAMPEÓN</div>
    <?php endif; ?>
</div>




                                        </div>
                                        <div class="mfb-scores">
                                            <?php if ( null !== $sr_home ) : ?>
                                                <span class="<?php echo esc_attr( $score_class_main ); ?>"><?php echo esc_html( $sr_home ); ?></span>
                                            <?php endif; ?>
                                            <?php if ( $went_penalties && null !== $pen_home ) : ?>
                                                <span class="<?php echo esc_attr( $score_class_pen ); ?>">(<?php echo esc_html( $pen_home ); ?>)</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>

                                    <div class="mfb-team-row <?php echo $away_is_win ? 'mfb-winner' : ''; ?>">
                                        <div class="mfb-team">
                                            <div class="mfb-team-logo">
                                                <?php if ( $away_team && ! empty( $away_team['logo'] ) ) : ?>
                                                    <img src="<?php echo esc_url( $away_team['logo'] ); ?>" alt="<?php echo esc_attr( $away_team['name'] ); ?>">
                                                <?php endif; ?>
                                            </div>
                                            <div class="mfb-team-name">
    <?php echo esc_html( $away_team ? $away_team['name'] : 'Por definir' ); ?>

    <?php if ( ! empty( $scorers_away_lines ) ) : ?>
        <div class="mfb-scorers-block">
            <?php foreach ( $scorers_away_lines as $line ) : ?>
                <div class="mfb-scorers-line">
                    <span class="mfb-scorer-ball">⚽</span>
                    <span class="mfb-scorer-text"><?php echo esc_html( $line ); ?></span>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <?php if ( $is_final_round && $finished && $away_is_win ) : ?>
        <div class="mfb-champion-label">🏆 CAMPEÓN</div>
    <?php endif; ?>
</div>






                                        </div>
                                        <div class="mfb-scores">
                                            <?php if ( null !== $sr_away ) : ?>
                                                <span class="<?php echo esc_attr( $score_class_main ); ?>"><?php echo esc_html( $sr_away ); ?></span>
                                            <?php endif; ?>
                                            <?php if ( $went_penalties && null !== $pen_away ) : ?>
                                                <span class="<?php echo esc_attr( $score_class_pen ); ?>">(<?php echo esc_html( $pen_away ); ?>)</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>

                                    <?php if ( $went_penalties ) : ?>
                                        <div class="mfb-pen-label">Decidido por penales</div>
                                    <?php endif; ?>

                                    <?php if ( $date_time_str || $stadium_name || $is_live ) : ?>
                                        <div class="mfb-match-footer">
                                            <?php if ( $date_time_str || $stadium_name ) : ?>
                                                <div class="mfb-match-meta">
                                                    <?php if ( $date_time_str ) : ?>
                                                        <div><?php echo esc_html( $date_time_str ); ?></div>
                                                    <?php endif; ?>
                                                    <?php if ( $stadium_name ) : ?>
                                                        <div><?php echo esc_html( $stadium_name ); ?></div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php endif; ?>
                                            <?php if ( $is_live ) : ?>
                                                <span class="mfb-live-badge">¡EN VIVO!</span>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /* =========================
     *  Shortcode + AJAX
     * =======================*/
    
    /**
     * Calcula los ganadores de cada partido a partir de la estructura de rondas.
     * Devuelve un arreglo asociativo id_partido => datos_equipo (id, name, logo).
     */
    private function calculate_winners_from_rounds( $rounds ) {
        $winners = array();

        if ( ! is_array( $rounds ) ) {
            return $winners;
        }

        foreach ( $rounds as $round ) {
            if ( empty( $round['matches'] ) || ! is_array( $round['matches'] ) ) {
                continue;
            }

            foreach ( $round['matches'] as $match ) {
                $home_team = null;
                $away_team = null;

                if ( isset( $match['home_from'] ) && isset( $winners[ $match['home_from'] ] ) ) {
                    $home_team = $winners[ $match['home_from'] ];
                } elseif ( isset( $match['home_team_id'] ) ) {
                    $home_team = $this->get_team_data( $match['home_team_id'] );
                }

                if ( isset( $match['away_from'] ) && isset( $winners[ $match['away_from'] ] ) ) {
                    $away_team = $winners[ $match['away_from'] ];
                } elseif ( isset( $match['away_team_id'] ) ) {
                    $away_team = $this->get_team_data( $match['away_team_id'] );
                }

                $match_id = isset( $match['id'] ) ? $match['id'] : null;

                $sr_home = isset( $match['score_regular']['home'] ) ? intval( $match['score_regular']['home'] ) : null;
                $sr_away = isset( $match['score_regular']['away'] ) ? intval( $match['score_regular']['away'] ) : null;

                $went_penalties = ! empty( $match['went_penalties'] );
                $pen_home       = isset( $match['penalties']['home'] ) ? intval( $match['penalties']['home'] ) : null;
                $pen_away       = isset( $match['penalties']['away'] ) ? intval( $match['penalties']['away'] ) : null;

                $finished = ! empty( $match['finished'] );

                $winner = null;

                if ( $finished && $home_team && $away_team && null !== $sr_home && null !== $sr_away ) {
                    if ( $sr_home > $sr_away ) {
                        $winner = $home_team;
                    } elseif ( $sr_away > $sr_home ) {
                        $winner = $away_team;
                    } elseif ( $went_penalties && null !== $pen_home && null !== $pen_away ) {
                        if ( $pen_home > $pen_away ) {
                            $winner = $home_team;
                        } elseif ( $pen_away > $pen_home ) {
                            $winner = $away_team;
                        }
                    }
                }

                if ( $match_id && $winner ) {
                    $winners[ $match_id ] = $winner;
                }
            }
        }

        return $winners;
    }

public function shortcode_bracket( $atts ) {
        $atts = shortcode_atts(
            array(
                'id' => 0,
            ),
            $atts,
            'football_bracket'
        );

        $post_id = intval( $atts['id'] );
        if ( ! $post_id ) {
            return '<p style="color:red;">Falta el atributo id en el shortcode.</p>';
        }

        wp_enqueue_style( 'mfb-bracket-style' );
        wp_enqueue_script( 'mfb-bracket-live' );

        $title = get_the_title( $post_id );

        ob_start();
        ?>
        <div class="mfb-bracket-wrapper" data-bracket-id="<?php echo esc_attr( $post_id ); ?>">
            <?php if ( $title ) : ?>
                <h2 class="mfb-tournament-title"><?php echo esc_html( $title ); ?></h2>
            <?php endif; ?>

            <?php echo $this->render_bracket_html( $post_id ); ?>
        </div>
        <?php
        return ob_get_clean();
    }

    public function ajax_get_bracket_html() {
        $post_id = isset( $_POST['id'] ) ? intval( $_POST['id'] ) : 0;
        if ( ! $post_id ) {
            wp_die( 'ID inválido' );
        }

        echo $this->render_bracket_html( $post_id );
        wp_die();
    }
}

new MFB_Live_Panel_Ajax_V7();
